//
//  QSICommand.m
//  QSICamera
//
//  Copyright (c) 2011, 2012, Joe Shimkus
//   All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//  	• 	Redistributions of source code must retain the above copyright
//      	notice, this list of conditions and the following disclaimer.
//  	• 	Redistributions in binary form must reproduce the above copyright
//        notice, this list of conditions and the following disclaimer in the
//        documentation and/or other materials provided with the distribution.
//  	• 	The name of Joe Shimkus may not be used to endorse or promote
//        products derived from this software without specific prior written
//        permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
//  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
//  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//  PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
//  OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
//  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
//  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
//  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
//  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
//  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "QSICommandProtected.h"

#import "QSIBigEndianData.h"

/***********************************************************************************************
************************************************************************************************
* QSICommand Private Methods
************************************************************************************************
***********************************************************************************************/

//
// QSICommand Private Class Methods
//
@interface QSICommand (QSICommandPrivateClassMethods)
@end // @interface QSICommand (QSICommandPrivateClassMethods)

@implementation QSICommand (QSICommandPrivateClassMethods)
@end // @implementation QSICommand (QSICommandPrivateClassMethods)

//
// QSICommand Private Instance Methods
//
@interface QSICommand (QSICommandPrivateInstanceMethods)
- (QSIStatus) receiveResponse : (char *) aBuffer
                     ofLength : (uint8_t *) aLength;
- (QSIStatus) sendCommand : (char *) aBuffer
                 ofLength : (uint8_t *) aLength;
@end // @interface QSICommand (QSICommandPrivateInstanceMethods)

@implementation QSICommand (QSICommandPrivateInstanceMethods)
- (QSIStatus) receiveResponse : (char *) aBuffer
                     ofLength : (uint8_t *) aLength
{
  QSIStatus status  = QSISuccess;

  QSIResponseBufferHeader *responseHeader = (QSIResponseBufferHeader *) aBuffer;

  uint32_t  numberOfBytes = *aLength;

  status = [_commsObject readToBuffer:aBuffer
                        numberOfBytes:&numberOfBytes];
  if ((status == QSISuccess) && (numberOfBytes != (*aLength)))
  {
    QSILog(@"receiving response returned incorrect length; expected: %d, received: %d", *aLength, numberOfBytes);

    status = QSICommandFailed;
  }

  // Verify the response is to the command we sent.
  if ((status == QSISuccess) && (responseHeader->_command != [self commandId]))
  {
    QSILog(@"received response did not match sent command; expected: 0x%x, received: 0x%x",
           [self commandId],
           responseHeader->_command);

    status = QSICommandFailed;
  }

  return (status);
} // end -receiveResponse:ofLength:

- (QSIStatus) sendCommand : (char *) aBuffer
                 ofLength : (uint8_t *) aLength
{
  QSIStatus status  = QSISuccess;

  uint32_t  numberOfBytes = *aLength;

  status = [_commsObject writeFromBuffer:aBuffer numberOfBytes:&numberOfBytes];

  return (status);
} // end -sendCommand:ofLength:
@end // @implementation QSICommand (QSICommandPrivateInstanceMethods)


/***********************************************************************************************
************************************************************************************************
* QSICommand Infrastructure Methods
************************************************************************************************
***********************************************************************************************/

//
// QSICommand Infrastructure Class Methods
//
@implementation QSICommand (QSICommandInfrastructureClassMethods)
@end // @implementation QSICommand (QSICommandInfrastructureClassMethods)

//
// QSICommand Infrastructure Instance Methods
//
@implementation QSICommand (QSICommandInfrastructureInstanceMethods)
- (void) dealloc
{
  [_endianTranslator release];
  [_commsObject release];

  if (_commandBuffer != NULL)
  {
    free(_commandBuffer);
  }

  if (_responseBuffer != NULL)
  {
    free(_responseBuffer);
  }

  [super dealloc];
} // end -dealloc

- (id) init
{
  [self doesNotRecognizeSelector:_cmd];

  return (nil);
} // end -init
@end // @implementation QSICommand (QSICommandInfrastructureInstanceMethods)

/***********************************************************************************************
************************************************************************************************
* QSICommand Overridden Methods
************************************************************************************************
***********************************************************************************************/

//
// QSICommand Overridden Class Methods
//
@implementation QSICommand (QSICommandOverriddenClassMethods)
@end // @implementation QSICommand (QSICommandOverriddenClassMethods)

//
// QSICommand Overridden Instance Methods
//
@implementation QSICommand (QSICommandOverriddenInstanceMethods)
@end // @implementation QSICommand (QSICommandOverriddenInstanceMethods)


/***********************************************************************************************
************************************************************************************************
* QSICommand Public Methods
************************************************************************************************
***********************************************************************************************/

//
// QSICommand Public Class Methods
//
@implementation QSICommand (QSICommandPublicClassMethods)
@end // @implementation QSICommand (QSICommandPublicClassMethods)

//
// QSICommand Public Instance Methods
//
@implementation QSICommand (QSICommandPublicInstanceMethods)
- (QSIStatus) execute
{
  QSIStatus status  = QSISuccess;

  // Preprocess the command buffer.
  [self preprocessCommandBuffer];

  // Send the command.
  uint8_t transmitLength  = _commandBufferLength;

  status = [self sendCommand:[self commandBuffer]
                    ofLength:&transmitLength];
  if (status == QSISuccess)
  {
    uint8_t receiveLength = _responseBufferLength;

    status = [self receiveResponse:[self responseBuffer]
                          ofLength:&receiveLength];
  }

  // Postprocess the response buffer.
  [self postprocessResponseBuffer];

  // If the comms succeeded check the command for success.
  if (status == QSISuccess)
  {
    QSIResponseBufferTrailer  *trailer;

    trailer = (QSIResponseBufferTrailer *) ([self responseBuffer] + _responseBufferLength - sizeof(*trailer));

    if (trailer->_error != 0)
    {
      status = QSIFailure;
    }
  }

  return (status);
} // end -execute

- (uint8_t) attemptLimit
{
  return (1);
} // end -attemptLimit
@end // @implementation QSICommand (QSICommandPublicInstanceMethods)

/***********************************************************************************************
************************************************************************************************
* QSICommand Protected Methods
************************************************************************************************
***********************************************************************************************/

//
// QSICommand Protected Class Methods
//
@implementation QSICommand (QSICommandProtectedClassMethods)
@end // @implementation QSICommand (QSICommandProtectedClassMethods)

//
// QSICommand Protected Instance Methods
//
@implementation QSICommand (QSICommandProtectedInstanceMethods)
- (id) initUsingComms : (QSIComms *) aCommsObject
         andCommandId : (QSICommandId)  aCommandId
{
  return ([self initUsingComms:aCommsObject
                     commandId:aCommandId
          andCommandBufferSize:sizeof(QSICommandBuffer)]);
} // end -initUsingComms:andQSICommandId:

- (id) initUsingComms : (QSIComms *) aCommsObject
            commandId : (QSICommandId)  aCommandId
 andCommandBufferSize : (uint8_t) aCommandBufferSize
{
  return ([self initUsingComms:aCommsObject
                     commandId:aCommandId
             commandBufferSize:aCommandBufferSize
         andResponseBufferSize:sizeof(QSIResponseBuffer)]);
} // end -initUsingComms:commandId:andQSICommandBufferSize:

- (id) initUsingComms : (QSIComms *) aCommsObject
            commandId : (QSICommandId) aCommandId
andResponseBufferSize : (uint8_t) aResponseBufferSize
{
  return ([self initUsingComms:aCommsObject
                     commandId:aCommandId
             commandBufferSize:sizeof(QSICommandBuffer)
         andResponseBufferSize:aResponseBufferSize]);
} // end -initUsingComms:commandId:andResponseBufferSize:

- (id) initUsingComms : (QSIComms *) aCommsObject
            commandId : (QSICommandId)  aCommandId
    commandBufferSize : (uint8_t) aCommandBufferSize
andResponseBufferSize : (uint8_t) aResponseBufferSize
{
  self = [super init];
  if (self != nil)
  {
    _commsObject  = [aCommsObject retain];

    _endianTranslator = [[QSIBigEndianData alloc] init];

    _commandBufferLength  = aCommandBufferSize;
    _responseBufferLength = aResponseBufferSize;

    _commandBuffer  = malloc(_commandBufferLength);
    _responseBuffer = malloc(_responseBufferLength);

    if ((_endianTranslator == nil) || (_commandBuffer == NULL) || (_responseBuffer == NULL))
    {
      [self release];
      self = nil;
    }
  }

  if (self != nil)
  {
    ((QSICommandBuffer *) _commandBuffer)->_header._command        = aCommandId;
    ((QSICommandBuffer *) _commandBuffer)->_header._commandLength  = _commandBufferLength - sizeof(QSICommandBuffer);
  }

  return (self);
} // end -initUsingComms:commandId:commandBufferSize:andResponseBufferSize:

- (char *) commandBuffer
{
  return (_commandBuffer);
} // end -commandBuffer

- (QSICommandId) commandId
{
  QSICommandBuffer  *buffer;

  buffer = (QSICommandBuffer *) [self commandBuffer];

  return (buffer->_header._command);
} // end -commandId

- (char *) responseBuffer
{
  return (_responseBuffer);
} // end -responseBuffer

//
// The following are used for setting correct byte ordering.
//
- (void) preprocessCommandBuffer
{
} // end -preprocessCommandBuffer

- (void) postprocessResponseBuffer
{
} // end -postprocessResponseBuffer

- (QSIEndian *) endianTranslator
{
  return (_endianTranslator);
} // end -endianTranslator
@end // @implementation QSICommand (QSICommandProtectedInstanceMethods)

/***********************************************************************************************
************************************************************************************************
* The QSICommand
************************************************************************************************
***********************************************************************************************/

@implementation QSICommand
@end // @implementation QSICommand
