//
//  QSIComms.m
//  QSICamera
//
//  Copyright (c) 2011, 2012, Joe Shimkus
//   All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//  	• 	Redistributions of source code must retain the above copyright
//      	notice, this list of conditions and the following disclaimer.
//  	• 	Redistributions in binary form must reproduce the above copyright
//        notice, this list of conditions and the following disclaimer in the
//        documentation and/or other materials provided with the distribution.
//  	• 	The name of Joe Shimkus may not be used to endorse or promote
//        products derived from this software without specific prior written
//        permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
//  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
//  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//  PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
//  OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
//  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
//  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
//  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
//  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
//  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "QSICommsProtected.h"

#import "QSIUSBComms.h"

#import "QSICamera.h"

#import <IOKit/IOKitLib.h>

/***********************************************************************************************
************************************************************************************************
* QSIComms Private Methods
************************************************************************************************
***********************************************************************************************/

//
// QSIComms Private Class Methods
//
@interface QSIComms (QSICommsPrivateClassMethods)
+ (NSArray *) findUSBCameras;
@end // @interface QSIComms (QSICommsPrivateClassMethods)

@implementation QSIComms (QSICommsPrivateClassMethods)
+ (NSArray *) findUSBCameras
{
  return ([QSIUSBComms findAllCameras]);
} // end +findUSBCameras
@end // @implementation QSIComms (QSICommsPrivateClassMethods)

//
// QSIComms Private Instance Methods
//
@interface QSIComms (QSICommsPrivateInstanceMethods)
- (QSIStatus) setStandardReadTimeout : (uint16_t) aTimeout;
- (QSIStatus) setStandardWriteTimeout : (uint16_t) aTimeout;
@end // @interface QSIComms (QSICommsPrivateInstanceMethods)

@implementation QSIComms (QSICommsPrivateInstanceMethods)
- (QSIStatus) setStandardReadTimeout : (uint16_t) aTimeout
{
  _standardReadTimeout  = aTimeout;

  return (QSISuccess);
} // end -setStandardReadTimeout:

- (QSIStatus) setStandardWriteTimeout : (uint16_t) aTimeout
{
  _standardWriteTimeout = aTimeout;

  return (QSISuccess);
} // end -setStandardWriteTimeout:
@end // @implementation QSIComms (QSICommsPrivateInstanceMethods)


/***********************************************************************************************
************************************************************************************************
* QSIComms Infrastructure Methods
************************************************************************************************
***********************************************************************************************/

//
// QSIComms Infrastructure Class Methods
//
@implementation QSIComms (QSICommsInfrastructureClassMethods)
@end // @implementation QSIComms (QSICommsInfrastructureClassMethods)

//
// QSIComms Infrastructure Instance Methods
//
@implementation QSIComms (QSICommsInfrastructureInstanceMethods)
- (void) dealloc
{
  (*_deviceInterface)->Release(_deviceInterface);

  [super dealloc];
} // end -dealloc

- (id) init
{
  [self doesNotRecognizeSelector:_cmd];

  return (nil);
} // end -init
@end // @implementation QSIComms (QSICommsInfrastructureInstanceMethods)

/***********************************************************************************************
************************************************************************************************
* QSIComms Overridden Methods
************************************************************************************************
***********************************************************************************************/

//
// QSIComms Overridden Class Methods
//
@implementation QSIComms (QSICommsOverriddenClassMethods)
@end // @implementation QSIComms (QSICommsOverriddenClassMethods)

//
// QSIComms Overridden Instance Methods
//
@implementation QSIComms (QSICommsOverriddenInstanceMethods)
@end // @implementation QSIComms (QSICommsOverriddenInstanceMethods)


/***********************************************************************************************
************************************************************************************************
* QSIComms Public Methods
************************************************************************************************
***********************************************************************************************/

//
// QSIComms Public Class Methods
//
@implementation QSIComms (QSICommsPublicClassMethods)
+ (NSArray *) findAllCameras
{
  QSIDebugLog(@"looking for cameras");

  return ([self findUSBCameras]);
} // end +findAllCameras
@end // @implementation QSIComms (QSICommsPublicClassMethods)

//
// QSIComms Public Instance Methods
//
@implementation QSIComms (QSICommsPublicInstanceMethods)
- (QSIStatus) connect
{
  [self doesNotRecognizeSelector:_cmd];

  return (QSIFailure);
} // end -connect

- (uint16_t) extendedReadTimeout
{
  return (_extendedReadTimeout);
} // end -extendedReadTimeout

- (uint16_t) extendedWriteTimeout
{
  return (_extendedWriteTimeout);
} // end -extendedWriteTimeout

- (QSIStatus) purge
{
  [self doesNotRecognizeSelector:_cmd];

  return (QSIFailure);
} // end -purge

- (QSIStatus) readToBuffer : (void *) aBuffer
             numberOfBytes : (uint32_t *) aNumberOfBytes
{
  [self doesNotRecognizeSelector:_cmd];

  return (QSIFailure);
} // end -readToBuffer:numberOfBytes:

- (uint16_t) shortReadTimeout
{
  return (_shortReadTimeout);
} // end -shortReadTimeout

- (uint16_t) shortWriteTimeout
{
  return (_shortWriteTimeout);
} // end -shortWriteTimeout

- (QSIStatus) setReadTimeout : (uint16_t) aReadTimeout
             andWriteTimeout : (uint16_t) aWriteTimeout
{
  QSIStatus status  = QSISuccess;

  if ((_currentReadTimeout != aReadTimeout) || (_currentWriteTimeout != aWriteTimeout))
  {
    status = [self performSetOfReadTimeout:aReadTimeout andWriteTimeout:aWriteTimeout];
  }

  if (status == QSISuccess)
  {
    _currentReadTimeout   = aReadTimeout;
    _currentWriteTimeout  = aWriteTimeout;
  }

  return (status);
} // -setReadTimeout:andWriteTimeout:

- (QSIStatus) setToExtendedTimeouts
{
  return ([self setReadTimeout:_extendedReadTimeout andWriteTimeout:_extendedWriteTimeout]);
} // end -setToExtendedTimeouts

- (QSIStatus) setToShortTimeouts
{
  return ([self setReadTimeout:_shortReadTimeout andWriteTimeout:_shortWriteTimeout]);
} // end -setToShortTimeout

- (QSIStatus) setToStandardTimeouts
{
  return ([self setReadTimeout:_standardReadTimeout andWriteTimeout:_standardWriteTimeout]);
} // end -setToStandardTimeouts

- (uint16_t) standardReadTimeout
{
  return (_standardReadTimeout);
} // end -standardReadTimeout

- (uint16_t) standardWriteTimeout
{
  return (_standardWriteTimeout);
} // end -standardWriteTimeout

- (QSIStatus) writeFromBuffer : (void *) aBuffer
                numberOfBytes : (uint32_t *) aNumberOfBytes
{
  [self doesNotRecognizeSelector:_cmd];

  return (QSIFailure);
} // end -writeFromBuffer:numberOfBytes:
@end // @implementation QSIComms (QSICommsPublicInstanceMethods)

/***********************************************************************************************
************************************************************************************************
* QSIComms Protected Methods
************************************************************************************************
***********************************************************************************************/

//
// QSIComms Protected Class Methods
//
@implementation QSIComms (QSICommsProtectedClassMethods)
+ (Class) cameraClassFromDeviceInterface : (IUnknownVTbl * *) aDeviceInterface
{
  [self doesNotRecognizeSelector:_cmd];

  return (nil);
} // end +cameraClassFromDeviceInterface:

+ (id) cameraUsingDeviceInterface : (IUnknownVTbl * *) aDeviceInterface
{
  QSIDebugLog(@"instantiating camera using device interface: %p", aDeviceInterface);

  Class     cameraClass;

  QSICamera *camera = nil;

  cameraClass = [self cameraClassFromDeviceInterface:aDeviceInterface];
  if (cameraClass != nil)
  {
    QSIComms  *comms  = nil;

    comms = [[[self alloc] initWithDeviceInterface:aDeviceInterface
                                  shortReadTimeout:[cameraClass shortReadTimeout]
                                 shortWriteTimeout:[cameraClass shortWriteTimeout]
                               standardReadTimeout:[cameraClass standardReadTimeout]
                              standardWriteTimeout:[cameraClass standardWriteTimeout]
                               extendedReadTimeout:[cameraClass extendedReadTimeout]
                           andExtendedWriteTimeout:[cameraClass extendedWriteTimeout]] autorelease];
    if (comms != nil)
    {
      camera = [[[cameraClass alloc] initWithComms:comms] autorelease];
    }
  }

  return (camera);
} // end +cameraUsingDeviceInterface:
@end // @implementation QSIComms (QSICommsProtectedClassMethods)

//
// QSIComms Protected Instance Methods
//
@implementation QSIComms (QSICommsProtectedInstanceMethods)
- (id) initWithDeviceInterface : (IUnknownVTbl * *) aDeviceInterface
              shortReadTimeout : (uint16_t) aShortReadTimeout
             shortWriteTimeout : (uint16_t) aShortWriteTimeout
           standardReadTimeout : (uint16_t) aStandardReadTimeout
          standardWriteTimeout : (uint16_t) aStandardWriteTimeout
           extendedReadTimeout : (uint16_t) anExtendedReadTimeout
       andExtendedWriteTimeout : (uint16_t) anExtendedWriteTimeout
{
  self = [super init];
  if (self != nil)
  {
    _currentReadTimeout   = -1;
    _currentWriteTimeout  = -1;

    _deviceInterface = aDeviceInterface;
    (*_deviceInterface)->AddRef(_deviceInterface);

    _extendedReadTimeout  = anExtendedReadTimeout;
    _extendedWriteTimeout = anExtendedWriteTimeout;

    _shortReadTimeout     = aShortReadTimeout;
    _shortWriteTimeout    = aShortWriteTimeout;

    [self setStandardReadTimeout:aStandardReadTimeout];
    [self setStandardWriteTimeout:aStandardWriteTimeout];

    [self setConnected:no];
  }

  return (self);
} // end -initWithStandardReadTimeout:standardWriteTimeout:extendedReadTimeout:andExtendedWriteTimeout:

- (IUnknownVTbl * *) deviceInterface
{
  return (_deviceInterface);
} // end -deviceInterface

- (QSIStatus) performSetOfReadTimeout : (uint16_t) aReadTimeout
                      andWriteTimeout : (uint16_t) aWriteTimeout
{
  [self doesNotRecognizeSelector:_cmd];

  return (QSIFailure);
} // end -performSetOfReadTimeout:andWriteTimeout:

- (uint16_t) readTimeout
{
  return (_currentReadTimeout);
} // end -readTimeout

- (void) setConnected : (bool) aIsConnected
{
  _isConnected = aIsConnected;
} // end -setConnected:

- (uint16_t) writeTimeout
{
  return (_currentWriteTimeout);
} // end -writeTimeout
@end // @implementation QSIComms (QSICommsProtectedInstanceMethods)

/***********************************************************************************************
************************************************************************************************
* The QSIComms
************************************************************************************************
***********************************************************************************************/

@implementation QSIComms
//
// The QSIComms protocol
//
- (QSIStatus) disconnect
{
  [self doesNotRecognizeSelector:_cmd];

  return (QSIFailure);
} // end -disconnect

- (bool) isConnected
{
  return (_isConnected);
} // end -isConnected
@end // @implementation QSIComms
